﻿Imports FormMain
Imports Peak.Can.IsoTp

''' <summary>
''' Form class handling configuration of an ISO-TP mapping.
''' </summary>
Public Class FormMapping
    ''' <summary>
    ''' The configured mapping.
    ''' </summary>
    Public m_mapping As FormMain.MappingStatus

    ''' <summary>
    ''' Default constructor
    ''' </summary>
    Public Sub New()
        InitializeComponent()
        m_mapping = New MappingStatus()
        ' fill target type combobox
        comboBoxTargetType.Items.Add(New ComboBoxItem(
            "PCANTP_ADDRESSING_PHYSICAL",
            TPCANTPAddressingType.PCANTP_ADDRESSING_PHYSICAL))
        comboBoxTargetType.Items.Add(New ComboBoxItem(
            "PCANTP_ADDRESSING_FUNCTIONAL",
            TPCANTPAddressingType.PCANTP_ADDRESSING_FUNCTIONAL))
        ' fill message type combobox
        fillComboBoxMsgType(m_mapping.m_canIdType)
        ' fill format type combobox
        fillComboBoxFormatType(m_mapping.m_canIdType, m_mapping.m_msgType)
        buttonOk.Enabled = checkMapping()
    End Sub
    ''' <summary>
    ''' Initializes the components with a pre-defined ISO-TP mapping
    ''' </summary>
    ''' <param name="mapping">The mapping used to initialize UI components (null for default settigns).</param>
    Public Sub initializeMapping(Optional ByVal mapping As MappingStatus = Nothing)
        Dim i As Integer
        Dim cbItem As ComboBoxItem

        If mapping Is Nothing Then
            mapping = New MappingStatus()
        End If
        ' select CAN ID Type in corresponding combobox
        Select Case mapping.m_canIdType
            Case TPCANTPIdType.PCANTP_ID_CAN_29BIT
                radioButtonCanId11b.Checked = False
                radioButtonCanId29b.Checked = True
                Exit Select
            Case TPCANTPIdType.PCANTP_ID_CAN_11BIT
                radioButtonCanId11b.Checked = True
                radioButtonCanId29b.Checked = False
                Exit Select
        End Select
        ' select message type in corresponding combobox
        For i = 0 To comboBoxMsgType.Items.Count - 1
            cbItem = DirectCast(comboBoxMsgType.Items(i), ComboBoxItem)
            If DirectCast(cbItem.Data, TPCANTPMessageType) = mapping.m_msgType Then
                comboBoxMsgType.SelectedIndex = i
                Exit For
            End If
        Next
        ' select format type in corresponding combobox
        For i = 0 To comboBoxFormat.Items.Count - 1
            cbItem = DirectCast(comboBoxFormat.Items(i), ComboBoxItem)
            If DirectCast(cbItem.Data, TPCANTPFormatType) = mapping.m_formatType Then
                comboBoxFormat.SelectedIndex = i
                Exit For
            End If
        Next
        ' select target type in corresponding combobox
        For i = 0 To comboBoxTargetType.Items.Count - 1
            cbItem = DirectCast(comboBoxTargetType.Items(i), ComboBoxItem)
            If DirectCast(cbItem.Data, TPCANTPAddressingType) = mapping.m_targetType Then
                comboBoxTargetType.SelectedIndex = i
                Exit For
            End If
        Next
        ' set CAN ID & CAN ID response
        numericUpDownCanId.Value = mapping.m_canId
        numericUpDownCanIdResp.Value = mapping.m_canIdResponse
        ' set addresses
        numericUpDownSourceAddr.Value = mapping.m_sourceAddr
        numericUpDownTargetAddr.Value = mapping.m_targetAddr
        numericUpDownRemoteAddr.Value = mapping.m_remoteAddr
        ' update UI components based on selected values
        updateUi()
    End Sub

    ''' <summary>
    ''' Fills the comboBox "Message Type" based on pre-selected inputs.
    ''' </summary>
    ''' <param name="idType">CAN ID Type of the mapping.</param>
    Private Sub fillComboBoxMsgType(ByVal idType As TPCANTPIdType)
        Dim cbItem As ComboBoxItem = Nothing

        If comboBoxFormat.SelectedItem Is Nothing Then
            cbItem = Nothing
        Else
            cbItem = DirectCast(comboBoxFormat.SelectedItem, ComboBoxItem)
        End If
        comboBoxFormat.Items.Clear()
        ' fill message type combobox
        comboBoxMsgType.Items.Add(New ComboBoxItem(
            "PCANTP_MESSAGE_DIAGNOSTIC",
            TPCANTPMessageType.PCANTP_MESSAGE_DIAGNOSTIC))
        If idType = TPCANTPIdType.PCANTP_ID_CAN_11BIT Then
            comboBoxMsgType.Items.Add(New ComboBoxItem(
                "PCANTP_MESSAGE_REMOTE_DIAGNOSTIC",
                TPCANTPMessageType.PCANTP_MESSAGE_REMOTE_DIAGNOSTIC))
        End If
        ' reminder: PCANTP_FORMAT_ENHANCED (29bits only) does NOT require mapping
        comboBoxFormat.SelectedItem = cbItem
    End Sub
    ''' <summary>
    ''' Fills the comboBox "Format Type" based on pre-selected inputs.
    ''' </summary>
    ''' <param name="idType">CAN ID Type of the mapping.</param>
    ''' <param name="msgType">Message Type of the mapping.</param>
    Private Sub fillComboBoxFormatType(ByVal idType As TPCANTPIdType, ByVal msgType As TPCANTPMessageType)
        Dim cbItem As ComboBoxItem = Nothing

        ' Saves selection if any.
        If comboBoxFormat.SelectedItem Is Nothing Then
            cbItem = Nothing
        Else
            cbItem = DirectCast(comboBoxFormat.SelectedItem, ComboBoxItem)
        End If
        comboBoxFormat.Items.Clear()
        ' Fills format type combobox.
        Select Case msgType
            Case TPCANTPMessageType.PCANTP_MESSAGE_REMOTE_DIAGNOSTIC
                ' reminder: PCANTP_FORMAT_MIXED does NOT require mapping with 29 bits CAN ID
                If idType = TPCANTPIdType.PCANTP_ID_CAN_11BIT Then
                    comboBoxFormat.Items.Add(New ComboBoxItem(
                    "PCANTP_FORMAT_MIXED",
                    TPCANTPFormatType.PCANTP_FORMAT_MIXED))
                End If
                Exit Select
            Case TPCANTPMessageType.PCANTP_MESSAGE_DIAGNOSTIC
                comboBoxFormat.Items.Add(New ComboBoxItem(
                    "PCANTP_FORMAT_NORMAL",
                    TPCANTPFormatType.PCANTP_FORMAT_NORMAL))
                comboBoxFormat.Items.Add(New ComboBoxItem(
                    "PCANTP_FORMAT_EXTENDED",
                    TPCANTPFormatType.PCANTP_FORMAT_EXTENDED))
                ' reminder: PCANTP_FORMAT_FIXED_NORMAL does NOT require mapping
                Exit Select
        End Select
        ' reminder: PCANTP_FORMAT_ENHANCED (29bits only) does NOT require mapping
        comboBoxFormat.SelectedItem = cbItem
    End Sub

    ''' <summary>
    ''' Checks consistency of components selection.
    ''' </summary>
    Private Sub updateUi()
        ' note: we do not update component member "Enabled" directly
        ' to prevent flickering.
        Dim radioButtonCanId11b_Enabled As Boolean = True
        Dim radioButtonCanId29b_Enabled As Boolean = True
        Dim comboBoxMsgType_Enabled As Boolean = False
        Dim comboBoxFormat_Enabled As Boolean = False
        Dim comboBoxTargetType_Enabled As Boolean = False
        Dim numericUpDownCanId_Enabled As Boolean = False
        Dim numericUpDownCanIdResp_Enabled As Boolean = False
        Dim numericUpDownSourceAddr_Enabled As Boolean = False
        Dim numericUpDownTargetAddr_Enabled As Boolean = False
        Dim numericUpDownRemoteAddr_Enabled As Boolean = False

        ' ID type must be set to continue
        If Not radioButtonCanId11b.Checked AndAlso Not radioButtonCanId29b.Checked Then
            Return
        End If
        comboBoxMsgType_Enabled = True
        ' Message type must be set to continue
        If getSelectedMsgType() <> TPCANTPMessageType.PCANTP_MESSAGE_UNKNOWN Then
            Select Case getSelectedMsgType()
                Case TPCANTPMessageType.PCANTP_MESSAGE_REMOTE_DIAGNOSTIC
                    numericUpDownRemoteAddr_Enabled = True
                    Exit Select
                Case TPCANTPMessageType.PCANTP_MESSAGE_DIAGNOSTIC
                    ' do NOT enable remote address
                    numericUpDownRemoteAddr.Value = &H0
                    Exit Select
            End Select
            comboBoxFormat_Enabled = True
            ' Format type must be set to continue
            If getSelectedFormatType() <> TPCANTPFormatType.PCANTP_FORMAT_UNKNOWN Then
                comboBoxTargetType_Enabled = True
                ' Target type must be set to continue
                If getSelectedTargetType() <> TPCANTPAddressingType.PCANTP_ADDRESSING_UNKNOWN Then
                    Select Case getSelectedTargetType()
                        Case TPCANTPAddressingType.PCANTP_ADDRESSING_FUNCTIONAL
                            ' do NOT enable CAN ID response
                            Exit Select
                        Case TPCANTPAddressingType.PCANTP_ADDRESSING_PHYSICAL
                            numericUpDownCanIdResp_Enabled = True
                            Exit Select
                    End Select
                    numericUpDownCanId_Enabled = True
                    numericUpDownSourceAddr_Enabled = True
                    numericUpDownTargetAddr_Enabled = True
                End If
            End If
        End If
        Me.SuspendLayout()
        radioButtonCanId11b.Enabled = radioButtonCanId11b_Enabled
        radioButtonCanId29b.Enabled = radioButtonCanId29b_Enabled
        comboBoxMsgType.Enabled = comboBoxMsgType_Enabled
        comboBoxFormat.Enabled = comboBoxFormat_Enabled
        comboBoxTargetType.Enabled = comboBoxTargetType_Enabled
        numericUpDownCanId.Enabled = numericUpDownCanId_Enabled
        numericUpDownCanIdResp.Enabled = numericUpDownCanIdResp_Enabled
        numericUpDownSourceAddr.Enabled = numericUpDownSourceAddr_Enabled
        numericUpDownTargetAddr.Enabled = numericUpDownTargetAddr_Enabled
        numericUpDownRemoteAddr.Enabled = numericUpDownRemoteAddr_Enabled
        buttonOk.Enabled = checkMapping()
        Me.ResumeLayout()
    End Sub

    ''' <summary>
    ''' Chekcs that all the values for the mapping are filled in.
    ''' </summary>
    ''' <returns>true if the values are set, false otherwise</returns>
    Private Function checkMapping() As Boolean
        Dim canId As UInteger
        Dim canIdResp As UInteger
        Dim canIdType As TPCANTPIdType
        Dim format As TPCANTPFormatType
        Dim msgType As TPCANTPMessageType
        Dim targetType As TPCANTPAddressingType
        Dim sourceAddr As Byte
        Dim targetAddr As Byte
        Dim remoteAddr As Byte

        ' get mapping values
        Try
            canId = Convert.ToUInt32(numericUpDownCanId.Value)
            canIdResp = Convert.ToUInt32(numericUpDownCanIdResp.Value)
            canIdType = getSelectedIdType()
            msgType = getSelectedMsgType()
            format = getSelectedFormatType()
            targetType = getSelectedTargetType()
            sourceAddr = Convert.ToByte(numericUpDownSourceAddr.Value)
            targetAddr = Convert.ToByte(numericUpDownTargetAddr.Value)
            remoteAddr = Convert.ToByte(numericUpDownRemoteAddr.Value)
        Catch ex As Exception
            MessageBox.Show(ex.Message, My.Resources.AppName, MessageBoxButtons.OK, MessageBoxIcon.[Error])
            Return False
        End Try
        ' check comboboxes selection
        If msgType = TPCANTPMessageType.PCANTP_MESSAGE_UNKNOWN Then
            Return False
        End If
        If format = TPCANTPFormatType.PCANTP_FORMAT_UNKNOWN Then
            Return False
        End If
        If targetType = TPCANTPAddressingType.PCANTP_ADDRESSING_UNKNOWN Then
            Return False
        End If
        ' Functional addressing is a one way communication,
        ' there is no need to check the target 
        If targetType <> TPCANTPAddressingType.PCANTP_ADDRESSING_FUNCTIONAL Then
            ' source and target addresses should not be the same
            ' thus it is the same for the CAN IDs 
            If (sourceAddr = targetAddr OrElse canId = canIdResp) Then
                Return False
            End If
        End If
        Return True
    End Function

#Region "Event Handlers"
    ''' <summary>
    ''' Event handler called when button Ok Is clicked.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub buttonOk_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles buttonOk.Click
        ' get mapping values
        Try
            m_mapping.m_canId = Convert.ToUInt32(numericUpDownCanId.Value)
            m_mapping.m_canIdResponse = Convert.ToUInt32(numericUpDownCanIdResp.Value)
            m_mapping.m_canIdType = getSelectedIdType()
            m_mapping.m_msgType = getSelectedMsgType()
            m_mapping.m_formatType = getSelectedFormatType()
            m_mapping.m_targetType = getSelectedTargetType()
            m_mapping.m_sourceAddr = Convert.ToByte(numericUpDownSourceAddr.Value)
            m_mapping.m_targetAddr = Convert.ToByte(numericUpDownTargetAddr.Value)
            m_mapping.m_remoteAddr = Convert.ToByte(numericUpDownRemoteAddr.Value)
        Catch ex As Exception
            ' An error occurs, report problem and aborts Dialog closure.
            MessageBox.Show(ex.Message, My.Resources.Resources.AppName, MessageBoxButtons.OK, MessageBoxIcon.Error)
            Me.DialogResult = DialogResult.None
        End Try
    End Sub

    ''' <summary>
    ''' Event handler called when selectin in comboBox "Format Type" Is changed.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub comboBoxFormat_SelectedIndexChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles comboBoxFormat.SelectedIndexChanged
        ' Unless Format Mixed is selected, Remote Address is disabled.
        If getSelectedFormatType() = TPCANTPFormatType.PCANTP_FORMAT_MIXED Then
            numericUpDownRemoteAddr.Enabled = True
        Else
            numericUpDownRemoteAddr.Enabled = False
            numericUpDownRemoteAddr.Value = &H0
        End If
        ' update OK button
        uiComponent_Leave(Me, EventArgs.Empty)
    End Sub
    ''' <summary>
    ''' Event handler called when selectin in comboBox "Message Type" Is changed.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub comboBoxMsgType_SelectedIndexChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles comboBoxMsgType.SelectedIndexChanged
        ' Updates comboBox "Format Addressing Type" contents:
        '  available format type is dependant of the can id type and the message type
        fillComboBoxFormatType(getSelectedIdType(), getSelectedMsgType())
        ' update OK button
        uiComponent_Leave(Me, EventArgs.Empty)
    End Sub
    ''' <summary>
    ''' Event handler called when selectin in comboBox "Target Addressing Type" Is changed.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub comboBoxTargetType_SelectedIndexChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles comboBoxTargetType.SelectedIndexChanged
        ' Can Id Response is not used if Functional Addressing is selected.
        If getSelectedTargetType() = TPCANTPAddressingType.PCANTP_ADDRESSING_FUNCTIONAL Then
            numericUpDownCanIdResp.Maximum = CanTpApi.CAN_ID_NO_MAPPING
            numericUpDownCanIdResp.Value = CanTpApi.CAN_ID_NO_MAPPING
            numericUpDownCanIdResp.Enabled = False
        Else
            numericUpDownCanIdResp.Maximum = numericUpDownCanId.Maximum
            numericUpDownCanIdResp.Enabled = True
            numericUpDownTargetAddr.Enabled = True
        End If
        ' update OK button
        uiComponent_Leave(Me, EventArgs.Empty)
    End Sub

    ''' <summary>
    ''' Event handler called when value Is changed in various numericUpDown components.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub numericUpDownComponents_ValueChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles numericUpDownCanIdResp.ValueChanged, numericUpDownTargetAddr.ValueChanged, numericUpDownSourceAddr.ValueChanged, numericUpDownRemoteAddr.ValueChanged, numericUpDownCanId.ValueChanged
        ' update OK button
        uiComponent_Leave(Me, EventArgs.Empty)
    End Sub
    ''' <summary>
    ''' Event handler called when radioButton "CAN ID Type" checked value changed.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub radioButtonCanId_CheckedChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles radioButtonCanId29b.CheckedChanged, radioButtonCanId11b.CheckedChanged
        ' update maximum CAN ID values
        If radioButtonCanId11b.Checked Then
            numericUpDownCanId.Maximum = &H7FF
        Else
            numericUpDownCanId.Maximum = &H1FFFFFFF
        End If
        numericUpDownCanIdResp.Maximum = numericUpDownCanId.Maximum
        ' available format type is dependant of the can id type and the message type
        fillComboBoxFormatType(getSelectedIdType(), getSelectedMsgType())
        ' update OK button
        uiComponent_Leave(Me, EventArgs.Empty)
    End Sub

    ''' <summary>
    ''' Event handler called when focus Is lost on various components.
    ''' </summary>
    ''' <param name="sender">The source of the event.</param>
    ''' <param name="e">Information on the event.</param>
    Private Sub uiComponent_Leave(ByVal sender As Object, ByVal e As EventArgs) Handles numericUpDownTargetAddr.Leave, numericUpDownSourceAddr.Leave, numericUpDownRemoteAddr.Leave, numericUpDownCanIdResp.Leave, numericUpDownCanId.Leave, comboBoxTargetType.Leave, comboBoxMsgType.Leave, comboBoxFormat.Leave
        updateUi()
    End Sub
#End Region

#Region "UI Getters"
    ''' <summary>
    ''' Returns the selected CAN ID Type.
    ''' </summary>
    ''' <returns>The selected CAN ID Type (PCANTP_ID_CAN_11BIT if none Is selected).</returns>
    Function getSelectedIdType() As TPCANTPIdType
        If radioButtonCanId29b.Checked Then
            Return TPCANTPIdType.PCANTP_ID_CAN_29BIT
        End If
        Return TPCANTPIdType.PCANTP_ID_CAN_11BIT
    End Function
    ''' <summary>
    ''' Returns the selected ISO-TP Message Type.
    ''' </summary>
    ''' <returns>The selected ISO-TP Message Type (PCANTP_MESSAGE_UNKNOWN if none Is selected).</returns>
    Function getSelectedMsgType() As TPCANTPMessageType
        If (comboBoxMsgType.SelectedItem Is Nothing) Then
            Return TPCANTPMessageType.PCANTP_MESSAGE_UNKNOWN
        End If
        Return DirectCast(DirectCast(comboBoxMsgType.SelectedItem, ComboBoxItem).Data, TPCANTPMessageType)
    End Function
    ''' <summary>
    ''' Returns the selected ISO-TP Format Type.
    ''' </summary>
    ''' <returns>The selected ISO-TP Format Type (PCANTP_FORMAT_UNKNOWN if none Is selected).</returns>
    Function getSelectedFormatType() As TPCANTPFormatType
        If (comboBoxFormat.SelectedItem Is Nothing) Then
            Return TPCANTPFormatType.PCANTP_FORMAT_UNKNOWN
        End If
        Return DirectCast(DirectCast(comboBoxFormat.SelectedItem, ComboBoxItem).Data, TPCANTPFormatType)
    End Function
    ''' <summary>
    ''' Returns the selected ISO-TP Target Addressing Type.
    ''' </summary>
    ''' <returns>The selected ISO-TP Target Addressing Type (PCANTP_ADDRESSING_UNKNOWN if none Is selected).</returns>
    Function getSelectedTargetType() As TPCANTPAddressingType
        If (comboBoxTargetType.SelectedItem Is Nothing) Then
            Return TPCANTPAddressingType.PCANTP_ADDRESSING_UNKNOWN
        End If
        Return DirectCast(DirectCast(comboBoxTargetType.SelectedItem, ComboBoxItem).Data, TPCANTPAddressingType)
    End Function
#End Region

End Class